<?php
// src/Models/Referral.php

class Referral {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    // 1. Link Users (On Signup)
    public function linkReferral($referrerCode, $newUserId) {
        $stmt = $this->pdo->prepare("SELECT id FROM users WHERE referral_code = ? AND is_active = 1");
        $stmt->execute([$referrerCode]);
        $referrerId = $stmt->fetchColumn();

        if ($referrerId && $referrerId != $newUserId) { // Prevent self-referral
            $stmt = $this->pdo->prepare("UPDATE users SET referred_by = ? WHERE id = ?");
            $stmt->execute([$referrerId, $newUserId]);
            return true;
        }
        return false;
    }

    // 2. Process Reward (On Order 'Delivered')
    public function processReward($orderId) {
        // Fetch Order & User Info
        $stmt = $this->pdo->prepare("
            SELECT o.id, o.user_id, o.order_number, u.referred_by 
            FROM orders o 
            JOIN users u ON o.user_id = u.id 
            WHERE o.id = ? AND o.order_status = 'delivered'
        ");
        $stmt->execute([$orderId]);
        $order = $stmt->fetch();

        if (!$order || !$order['referred_by']) return; // No referral link

        $referrerId = $order['referred_by'];
        $refereeId = $order['user_id'];

        // Rule: First Order Only?
        // Check if referee has ANY other delivered orders *before* this one
        $chk = $this->pdo->prepare("SELECT COUNT(*) FROM orders WHERE user_id = ? AND order_status = 'delivered' AND id != ?");
        $chk->execute([$refereeId, $orderId]);
        if ($chk->fetchColumn() > 0) return; // Not first order

        // Rule: Already Rewarded?
        $chkRew = $this->pdo->prepare("SELECT COUNT(*) FROM wallet_transactions WHERE reference_id = ? AND type = 'referral_bonus'");
        $chkRew->execute([$orderId]); // Use Order ID as unique ref key for specific bonus
        if ($chkRew->fetchColumn() > 0) return; // Already paid

        // Fetch Reward Amount (DB Settings or Config)
        // Hardcoded 50 for now, or fetch from settings table
        $amount = 50.00; 

        // Credit Wallet
        require_once __DIR__ . '/User.php';
        $userModel = new User($this->pdo);
        
        try {
            // Description
            $desc = "Referral Bonus for Order #" . $order['order_number'];
            $userModel->creditWallet($referrerId, $amount, 'referral_bonus', $orderId, $desc);
            
            // Log for Safety
            error_log("Referral Reward Processed: User $referrerId for Order $orderId");
        } catch (Exception $e) {
            error_log("Referral Reward Error: " . $e->getMessage());
        }
    }

    // 3. Process Reversal (On Order 'Returned' / 'Cancelled')
    public function processReversal($orderId) {
        // Check if this order generated a reward
        $stmt = $this->pdo->prepare("SELECT * FROM wallet_transactions WHERE reference_id = ? AND type = 'referral_bonus'");
        $stmt->execute([$orderId]);
        $txn = $stmt->fetch();

        if (!$txn) return; // No reward was given, so nothing to reverse

        $referrerId = $txn['user_id'];
        $amount = $txn['amount']; // The amount we gave

        // Debit Wallet
        require_once __DIR__ . '/User.php';
        $userModel = new User($this->pdo);
        
        // Note: creditWallet adds amount. So for reversal we pass NEGATIVE.
        $reverseAmount = -1 * abs($amount);
        $desc = "Referral Reversal - Order Returned #" . $orderId; 
        
        $userModel->creditWallet($referrerId, $reverseAmount, 'referral_reversal', $orderId, $desc);
    }

    // 4. Get User Referral Stats
    public function getStats($userId) {
        $stats = [
            'total_invites' => 0,
            'successful' => 0,
            'earned' => 0
        ];

        // Total Invites (People who signed up)
        $s1 = $this->pdo->prepare("SELECT COUNT(*) FROM users WHERE referred_by = ?");
        $s1->execute([$userId]);
        $stats['total_invites'] = $s1->fetchColumn();

        // Earned
        $s2 = $this->pdo->prepare("SELECT SUM(amount) FROM wallet_transactions WHERE user_id = ? AND type = 'referral_bonus'");
        $s2->execute([$userId]);
        $stats['earned'] = $s2->fetchColumn() ?: 0.00;

        // Successful (Count of bonuses)
        $s3 = $this->pdo->prepare("SELECT COUNT(*) FROM wallet_transactions WHERE user_id = ? AND type = 'referral_bonus'");
        $s3->execute([$userId]);
        $stats['successful'] = $s3->fetchColumn();

        return $stats;
    }

    // 5. Get Referral History list
    public function getHistory($userId) {
        // Get list of friends and their status
        $stmt = $this->pdo->prepare("
            SELECT u.full_name, u.created_at, 
            (SELECT COUNT(*) FROM orders o WHERE o.user_id = u.id AND o.order_status = 'delivered') as orders_count
            FROM users u 
            WHERE u.referred_by = ? 
            ORDER BY u.created_at DESC
        ");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
}
