<?php
// src/Models/Order.php

class Order {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    // 1. Create New Order (Main Transaction)
    public function createOrder($userId, $addressId, $paymentMethod, $useWallet = false) {
        try {
            $this->pdo->beginTransaction();

            // A. Fetch Cart Items & Verify Stock
            // Joining carts, cart_items, product_variants, products to get all pricing/stock info
            $sql = "SELECT ci.quantity, v.id as variant_id, v.stock_quantity, v.price_override, 
                           p.base_price, p.name 
                    FROM carts c
                    JOIN cart_items ci ON c.id = ci.cart_id
                    JOIN product_variants v ON ci.product_variant_id = v.id
                    JOIN products p ON v.product_id = p.id
                    WHERE c.user_id = ?";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$userId]);
            $cartItems = $stmt->fetchAll();

            if (empty($cartItems)) throw new Exception("Cart is empty");

            $totalAmount = 0;
            foreach ($cartItems as $item) {
                if ($item['stock_quantity'] < $item['quantity']) {
                    throw new Exception("Product {$item['name']} is out of stock.");
                }
                $price = $item['price_override'] ?: $item['base_price'];
                $totalAmount += $price * $item['quantity'];
            }

            // B. Wallet Deduction
            $walletUsed = 0;
            if ($useWallet) {
                $stmt = $this->pdo->prepare("SELECT wallet_balance FROM users WHERE id = ? FOR UPDATE"); // Lock User Row
                $stmt->execute([$userId]);
                $balance = $stmt->fetchColumn();
                
                if ($balance > 0) {
                    $walletUsed = min($balance, $totalAmount);
                    // Deduct
                    $stmt = $this->pdo->prepare("UPDATE users SET wallet_balance = wallet_balance - ? WHERE id = ?");
                    $stmt->execute([$walletUsed, $userId]);
                    // Log
                    $stmt = $this->pdo->prepare("INSERT INTO wallet_transactions (user_id, amount, type, description, created_at) VALUES (?, ?, 'purchase', ?, NOW())");
                    $stmt->execute([$userId, -$walletUsed, "Used for Order Payment"]);
                }
            }

            // C. Determine Status
            $payableAmount = $totalAmount - $walletUsed;
            // If fully paid by wallet, status is Paid. Else Pending (COD/Online).
            $paymentStatus = ($payableAmount <= 0) ? 'paid' : 'pending'; 
            if ($paymentMethod === 'cod' && $payableAmount > 0) $paymentStatus = 'pending';

            // D. Create Order Record
            $orderNumber = 'ORD-' . strtoupper(uniqid());
            
            // Fetch Address Snapshot (Simplified: Just taking ID implies we trust it exists, normally we verify)
            $stmt = $this->pdo->prepare("SELECT * FROM user_addresses WHERE id = ?");
            $stmt->execute([$addressId]);
            $addr = $stmt->fetch(PDO::FETCH_ASSOC);
            $addrJson = $addr ? json_encode($addr) : json_encode(['id' => $addressId]);

            $stmt = $this->pdo->prepare("INSERT INTO orders (order_number, user_id, total_amount, discount_amount, payment_method, payment_status, order_status, shipping_address_json, created_at) VALUES (?, ?, ?, ?, ?, ?, 'pending', ?, NOW())");
            // Here 'discount_amount' can be used to store wallet_amount used, or we add a column 'wallet_amount'. 
            // Reuse discount_amount for wallet usage for MVP simplicity.
            $stmt->execute([$orderNumber, $userId, $totalAmount, $walletUsed, $paymentMethod, $paymentStatus, $addrJson]);
            $orderId = $this->pdo->lastInsertId();

            // E. Insert Items & Updates Stock
            foreach ($cartItems as $item) {
                $price = $item['price_override'] ?: $item['base_price'];
                
                $stmt = $this->pdo->prepare("INSERT INTO order_items (order_id, product_variant_id, product_name_snapshot, price_at_purchase, quantity) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$orderId, $item['variant_id'], $item['name'], $price, $item['quantity']]);

                // Reduce Stock
                $stmt = $this->pdo->prepare("UPDATE product_variants SET stock_quantity = stock_quantity - ? WHERE id = ?");
                $stmt->execute([$item['quantity'], $item['variant_id']]);
            }

            // F. Clear Cart
            // Need Cart ID
            $stmt = $this->pdo->prepare("SELECT id FROM carts WHERE user_id = ?");
            $stmt->execute([$userId]);
            $cartId = $stmt->fetchColumn();
            if($cartId) {
                $this->pdo->prepare("DELETE FROM cart_items WHERE cart_id = ?")->execute([$cartId]);
            }

            // G. Timeline
            $this->pdo->prepare("INSERT INTO order_timeline (order_id, status, description, created_by, created_at) VALUES (?, 'pending', 'Order Placed', 'User', NOW())")->execute([$orderId]);

            $this->pdo->commit();
            return $orderNumber;

        } catch (Exception $e) {
            $this->pdo->rollBack();
            throw $e;
        }
    }

    // 2. Get User Orders
    public function getUserOrders($userId) {
        $stmt = $this->pdo->prepare("SELECT id, order_number, total_amount, order_status, created_at FROM orders WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }

    // 3. Get Details
    public function getOrderDetails($orderId, $userId) {
        $stmt = $this->pdo->prepare("SELECT * FROM orders WHERE (id = ? OR order_number = ?) AND user_id = ?");
        $stmt->execute([$orderId, $orderId, $userId]);
        $order = $stmt->fetch();

        if (!$order) return null;

        $stmt = $this->pdo->prepare("SELECT * FROM order_items WHERE order_id = ?");
        $stmt->execute([$order['id']]);
        $items = $stmt->fetchAll();

        $stmt = $this->pdo->prepare("SELECT * FROM order_timeline WHERE order_id = ? ORDER BY created_at ASC");
        $stmt->execute([$order['id']]);
        $timeline = $stmt->fetchAll();

        return ['info' => $order, 'items' => $items, 'timeline' => $timeline];
    }

    // 4. Cancel
    public function cancelOrder($userId, $orderId) { // Fixed Param Order relative to previous usage
        // Note: Logic copied from previous step but cleaned
        $orderInfo = $this->getOrderDetails($orderId, $userId);
        if (!$orderInfo) throw new Exception("Order not found.");
        $order = $orderInfo['info'];

        if (!in_array($order['order_status'], ['pending', 'confirmed', 'packed'])) throw new Exception("Cannot cancel status: " . $order['order_status']);

        try {
            $this->pdo->beginTransaction();

            $stmt = $this->pdo->prepare("UPDATE orders SET order_status = 'cancelled' WHERE id = ?");
            $stmt->execute([$order['id']]);

            $this->pdo->prepare("INSERT INTO order_timeline (order_id, status, description, created_by, created_at) VALUES (?, 'cancelled', 'Cancelled by User', 'User', NOW())")->execute([$order['id']]);

            // Restore Stock
            foreach ($orderInfo['items'] as $item) {
                $this->pdo->prepare("UPDATE product_variants SET stock_quantity = stock_quantity + ? WHERE id = ?")->execute([$item['quantity'], $item['product_variant_id']]);
            }

            // Refund if paid
            if ($order['payment_status'] === 'paid' && $order['total_amount'] > 0) {
                 require_once __DIR__ . '/User.php';
                 $userModel = new User($this->pdo);
                 $userModel->creditWallet($userId, $order['total_amount'], 'refund', $order['order_number'], "Refund for Order #" . $order['order_number']);
                 
                 $this->pdo->prepare("UPDATE orders SET payment_status = 'refunded' WHERE id = ?")->execute([$order['id']]);
            }

            $this->pdo->commit();
            return ['success' => true];
        } catch (Exception $e) {
            $this->pdo->rollBack();
            return ['error' => $e->getMessage()];
        }
    }

    // 5. Admin Update
    public function updateStatus($orderId, $newStatus, $adminId) {
        try {
            $this->pdo->beginTransaction();
            
            $stmt = $this->pdo->prepare("UPDATE orders SET order_status = ? WHERE id = ?");
            $stmt->execute([$newStatus, $orderId]);
            
            $this->pdo->prepare("INSERT INTO order_timeline (order_id, status, description, created_by, created_at) VALUES (?, ?, ?, 'Admin', NOW())")->execute([$orderId, $newStatus, "Status updated to $newStatus"]);

            if ($newStatus === 'delivered') {
                require_once __DIR__ . '/Referral.php';
                $refModel = new Referral($this->pdo);
                $refModel->processReward($orderId);
            }
            if ($newStatus === 'returned') {
                require_once __DIR__ . '/Referral.php';
                $refModel = new Referral($this->pdo);
                $refModel->processReversal($orderId);
            }

            $this->pdo->commit();
        } catch (Exception $e) {
            $this->pdo->rollBack();
            throw $e;
        }
    }
}
