<?php
// src/Models/Cart.php

class Cart {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getOrCreateCart($userId) {
        $stmt = $this->db->prepare("SELECT id FROM carts WHERE user_id = ?");
        $stmt->execute([$userId]);
        $cart = $stmt->fetch();

        if (!$cart) {
            $stmt = $this->db->prepare("INSERT INTO carts (user_id, created_at) VALUES (?, NOW())");
            $stmt->execute([$userId]);
            return $this->db->lastInsertId();
        }
        return $cart['id'];
    }

    public function add($userId, $variantId, $qty = 1) {
        $cartId = $this->getOrCreateCart($userId);

        // Check availability
        $vStmt = $this->db->prepare("SELECT stock_quantity, price_override, product_id FROM product_variants WHERE id = ?");
        $vStmt->execute([$variantId]);
        $variant = $vStmt->fetch();
        
        if (!$variant || $variant['stock_quantity'] < $qty) {
            return ['error' => 'Out of stock'];
        }

        // Check if exists in cart
        $check = $this->db->prepare("SELECT id, quantity FROM cart_items WHERE cart_id = ? AND product_variant_id = ?");
        $check->execute([$cartId, $variantId]);
        $existing = $check->fetch();

        if ($existing) {
            $newQty = $existing['quantity'] + $qty;
            if ($newQty > $variant['stock_quantity']) return ['error' => 'Max stock reached'];
            
            $update = $this->db->prepare("UPDATE cart_items SET quantity = ? WHERE id = ?");
            $update->execute([$newQty, $existing['id']]);
        } else {
            $insert = $this->db->prepare("INSERT INTO cart_items (cart_id, product_variant_id, quantity) VALUES (?, ?, ?)");
            $insert->execute([$cartId, $variantId, $qty]);
        }
        
        return ['success' => true];
    }

    public function get($userId) {
        $stmt = $this->db->prepare("SELECT ci.id as item_id, ci.quantity, 
                                           v.id as variant_id, v.color_name, v.size, v.price_override, 
                                           p.name, p.image_url, p.base_price, p.slug
                                    FROM carts c
                                    JOIN cart_items ci ON c.id = ci.cart_id
                                    JOIN product_variants v ON ci.product_variant_id = v.id
                                    JOIN products p ON v.product_id = p.id
                                    WHERE c.user_id = ?");
        $stmt->execute([$userId]);
        $items = $stmt->fetchAll();
        
        $total = 0;
        foreach ($items as &$item) {
            $item['price'] = $item['price_override'] ?: $item['base_price'];
            $item['line_total'] = $item['price'] * $item['quantity'];
            $total += $item['line_total'];
        }
        
        return ['items' => $items, 'total' => $total, 'count' => count($items)];
    }

    public function remove($userId, $itemId) {
        $cartId = $this->getOrCreateCart($userId);
        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE id = ? AND cart_id = ?");
        $stmt->execute([$itemId, $cartId]);
        return ['success' => true];
    }
    
    public function clear($userId) {
        $cartId = $this->getOrCreateCart($userId);
        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE cart_id = ?");
        $stmt->execute([$cartId]);
    }
}
