/**
 * Popup Manager for Android-Style Modals
 */
const Popup = {
    overlay: null,
    modal: null,
    title: null,
    message: null,
    icon: null,
    confirmBtn: null,
    cancelBtn: null,
    callback: null,

    init() {
        this.overlay = document.getElementById('popupOverlay');
        this.modal = document.getElementById('popupModal');
        this.title = document.getElementById('popupTitle');
        this.message = document.getElementById('popupMessage');
        this.icon = document.getElementById('popupIcon');
        this.confirmBtn = document.getElementById('popupConfirm');
        this.cancelBtn = document.getElementById('popupCancel');
        
        // Close on overlay click
        if(this.overlay) {
            this.overlay.addEventListener('click', (e) => {
                if(e.target === this.overlay) this.close();
            });
        }

        // Close on Escape
        document.addEventListener('keydown', (e) => {
            if(e.key === 'Escape') this.close();
        });
    },

    show(type, title, msg, onConfirm = null) {
        if (!this.overlay) this.init();

        // Set Content
        this.title.innerText = title;
        this.message.innerHTML = msg;
        this.callback = onConfirm;

        // Reset Styles
        this.icon.className = 'w-16 h-16 rounded-full flex items-center justify-center mb-4 mx-auto';
        this.confirmBtn.classList.remove('hidden');
        this.cancelBtn.classList.add('hidden');
        this.confirmBtn.innerText = 'Okay';
        this.confirmBtn.onclick = () => this.close();

        // Type Specifics
        if (type === 'success') {
            this.icon.innerHTML = '<svg class="w-8 h-8 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M5 13l4 4L19 7"></path></svg>';
            this.icon.classList.add('bg-green-100');
            this.confirmBtn.className = 'w-full py-3 rounded-xl font-bold text-white bg-green-500 shadow-lg shadow-green-200 active:scale-95 transition-transform';
        } else if (type === 'error') {
            this.icon.innerHTML = '<svg class="w-8 h-8 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M6 18L18 6M6 6l12 12"></path></svg>';
            this.icon.classList.add('bg-red-100');
            this.confirmBtn.className = 'w-full py-3 rounded-xl font-bold text-white bg-red-500 shadow-lg shadow-red-200 active:scale-95 transition-transform';
        } else if (type === 'confirm') {
            this.icon.innerHTML = '<svg class="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
            this.icon.classList.add('bg-blue-100');
            this.confirmBtn.innerText = 'Yes, Proceed';
            this.confirmBtn.className = 'flex-1 py-3 rounded-xl font-bold text-white bg-blue-600 shadow-lg shadow-blue-200 active:scale-95 transition-transform';
            this.cancelBtn.classList.remove('hidden');
            this.cancelBtn.onclick = () => this.close();
            
            // Override confirm click
            this.confirmBtn.onclick = () => {
                if (this.callback) this.callback();
                this.close();
            }
        }

        // Show
        this.overlay.classList.remove('hidden');
        requestAnimationFrame(() => {
            this.overlay.classList.remove('opacity-0');
            this.modal.classList.remove('scale-90', 'opacity-0');
        });
    },

    close() {
        if (!this.overlay) return;
        
        this.overlay.classList.add('opacity-0');
        this.modal.classList.add('scale-90', 'opacity-0');
        
        setTimeout(() => {
            this.overlay.classList.add('hidden');
        }, 300);
    }
};

document.addEventListener('DOMContentLoaded', () => Popup.init());
