<?php
// public_html/api/user/auth.php

require_once dirname(dirname(dirname(__DIR__))) . '/public_html/index.php'; // Bootstrap

// JSON Header
header('Content-Type: application/json');

// Check Request Method and AJAX
if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !is_ajax()) {
    json_response(['error' => 'Invalid Request', 'debug_svr' => $_SERVER], 400);
}

// Check Rate Limit (Generic IP check)
check_rate_limit('auth_api', 20, 60);

// CSRF Check
check_csrf();

$action = $_POST['scout_action'] ?? '';
$phone = sanitize_input($_POST['phone'] ?? '');

// Validation
if (empty($phone) || !preg_match('/^[0-9]{10}$/', $phone)) {
    json_response(['error' => 'Invalid Mobile Number'], 400);
}

// ACTION: SEND OTP
if ($action === 'send_otp') {
    $referralCode = sanitize_input($_POST['referral_code'] ?? '');
    
    try {
        // 1. Check if user exists
        $stmt = $pdo->prepare("SELECT id, is_active FROM users WHERE phone = ?");
        $stmt->execute([$phone]);
        $user = $stmt->fetch();

        // 2. If new user, create record (Inactive status)
        if (!$user) {
            $pdo->beginTransaction();
            
            // Generate own referral code
            $myRefCode = strtoupper(substr(md5(uniqid()), 0, 6)); 
            
            // Handle Referred By
            $referrerId = null;
            if (!empty($referralCode)) {
                $rStmt = $pdo->prepare("SELECT id FROM users WHERE referral_code = ?");
                $rStmt->execute([$referralCode]);
                $referrerId = $rStmt->fetchColumn();
            }

            $cStmt = $pdo->prepare("INSERT INTO users (phone, referral_code, referred_by, is_active) VALUES (?, ?, ?, 0)");
            $cStmt->execute([$phone, $myRefCode, $referrerId ?: null]);
            $userId = $pdo->lastInsertId();
            
            $pdo->commit();
        } else {
            $userId = $user['id'];
            if ($user['is_active'] == 0) {
                // Allow OTP for activation flow (optional check)
            }
        }

        // 3. Generate OTP
        $otp = (SITE_ENV === 'development') ? '123456' : rand(100000, 999999);
        $otpHash = password_hash($otp, PASSWORD_BCRYPT);
        $expiresAt = date('Y-m-d H:i:s', time() + 300); // 5 mins

        // 4. Store OTP Log
        $logStmt = $pdo->prepare("INSERT INTO otp_logs (phone, otp_hash, expires_at) VALUES (?, ?, ?)");
        $logStmt->execute([$phone, $otpHash, $expiresAt]);

        // 5. Send SMS (Simulated)
        if (SITE_ENV === 'development') {
            error_log("OTP for $phone is: $otp");
        } else {
            // TODO: Integrate Fast2SMS / Twilio here
            error_log("SMS Gateway Trigger: Sending $otp to $phone");
        }

        json_response(['success' => true, 'message' => 'OTP Sent']);

    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error("Auth Error (Send OTP): " . $e->getMessage());
        json_response(['error' => 'Server Error'], 500);
    }
}

// ACTION: VERIFY OTP
if ($action === 'verify_otp') {
    $otp = $_POST['otp'] ?? '';
    
    if (strlen($otp) !== 6) {
        json_response(['error' => 'Invalid OTP Format'], 400);
    }

    try {
        // 1. Fetch latest OTP
        $stmt = $pdo->prepare("SELECT * FROM otp_logs WHERE phone = ? AND is_verified = 0 ORDER BY id DESC LIMIT 1");
        $stmt->execute([$phone]);
        $otpRecord = $stmt->fetch();

        if (!$otpRecord) {
            json_response(['error' => 'OTP Expired or Not Found'], 400);
        }

        // 2. Validate
        if (strtotime($otpRecord['expires_at']) < time()) {
            json_response(['error' => 'OTP Expired'], 400);
        }

        if ($otpRecord['attempts'] >= 5) {
             json_response(['error' => 'Too many attempts. Request new OTP.'], 400);
        }

        if (!password_verify($otp, $otpRecord['otp_hash'])) {
            // Increment attempts
            $up = $pdo->prepare("UPDATE otp_logs SET attempts = attempts + 1 WHERE id = ?");
            $up->execute([$otpRecord['id']]);
            json_response(['error' => 'Incorrect OTP'], 400);
        }

        // 3. Success: Mark Verified
        $pdo->beginTransaction();

        $vStmt = $pdo->prepare("UPDATE otp_logs SET is_verified = 1 WHERE id = ?");
        $vStmt->execute([$otpRecord['id']]);

        // 4. Activate User & Login
        $uStmt = $pdo->prepare("SELECT * FROM users WHERE phone = ?");
        $uStmt->execute([$phone]);
        $user = $uStmt->fetch();

        if (!$user) throw new Exception("User not found after OTP check"); // Should not happen

        $currDate = date('Y-m-d H:i:s');
        $upUser = $pdo->prepare("UPDATE users SET is_active = 1, updated_at = ? WHERE id = ?");
        $upUser->execute([$currDate, $user['id']]);

        $pdo->commit();

        // 5. Session Setup
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_phone'] = $user['phone'];
        $_SESSION['user_name'] = $user['full_name'] ?? 'User';
        
        json_response(['success' => true]);

    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        log_error("Auth Error (Verify OTP): " . $e->getMessage());
        json_response(['error' => 'Verification Failed'], 500);
    }
}

json_response(['error' => 'Invalid Action'], 400);
