<?php
// includes/helpers.php

/**
 * HTML Escape (XSS Protection)
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Sanitize Input (Arrays or Strings)
 */
function sanitize_input($data) {
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            $data[$key] = sanitize_input($value);
        }
        return $data;
    }
    return strip_tags(trim($data));
}

/**
 * Format Price
 */
function format_price($amount) {
    return DEFAULT_CURRENCY . number_format((float)$amount, 2);
}

/**
 * Redirect Helper
 */
function redirect($path) {
    // If headers sent, use JS fallback
    if (headers_sent()) {
        echo "<script>window.location.href='" . BASE_URL . $path . "';</script>";
    } else {
        header("Location: " . BASE_URL . $path);
    }
    exit;
}

/**
 * JSON Response Helper
 */
function json_response($data = [], $status = 200) {
    header('Content-Type: application/json');
    http_response_code($status);
    echo json_encode($data);
    exit;
}

/**
 * Check if Request is AJAX
 */
function is_ajax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

/**
 * Get Current User (from Session)
 */
function current_user() {
    return $_SESSION['user'] ?? null;
}

/**
 * Log Error
 */
function log_error($message) {
    error_log("[" . date('Y-m-d H:i:s') . "] " . $message . PHP_EOL, 3, LOG_PATH . '/custom_errors.log');
}

/**
 * Rate Limiting (Helper)
 */
function check_rate_limit($action, $limit, $seconds) {
    if (!isset($_SESSION['rate_limit'][$action])) {
        $_SESSION['rate_limit'][$action] = ['count' => 0, 'start_time' => time()];
    }

    $data = &$_SESSION['rate_limit'][$action];

    // Reset
    if (time() - $data['start_time'] > $seconds) {
        $data['count'] = 0;
        $data['start_time'] = time();
    }

    $data['count']++;

    if ($data['count'] > $limit) {
        http_response_code(429);
        die("Too many requests. Please slow down.");
    }
}
